<?php
/*  =========================================================================
 *  ------------------------------ Shades.php -------------------------------
 *                                    v 0.75
/**
 *  A class with some helpful routines for handling color shades
 *
 *  Created by: Marian C. Buford
 *              mariancbuford.com (MCB)
 *  Created on: 06.10.2012
 *  License:    Beer-Ware
 *
 *  -------------------------------------------------------------------------
 *    "THE BEER-WARE LICENSE" (Revision 42):
 *    <marian[at]mariancbuford[dot]com> wrote this file. As long as you
 *    retain this notice you can do whatever you want with this stuff. If we
 *    meet some day, and you think this stuff is worth it, you can buy me a
 *    beer in return - Marian C. Buford
 *  ------------------------------------------------------------------------- *
 */
/*  ========================================================================= */
class Shades
{
     var $palette = array();

/*  =========================================================================
    CONSTRUCTOR
    ========================================================================= */
    function __construct($hex = '000000')
    {
        $this->setPalette($hex);
    } // CONSTRUCTOR

/*  =========================================================================
    DESTRUCTOR
    ========================================================================= */
    function __destruct() {
   } // DESTRUCTOR

/*  =========================================================================
    FUNCTION: setPalette()
    ========================================================================= */
    function setPalette($hex)
    {
        // Clear the existing palette
        $this->palette = array();

        // Make sure we got a valid hex value
        if (!$this->isHex($hex))
	{
            $this->raiseError('background color ' . $hex .
                            ' is not a hex color value',__FUNCTION__, __LINE__);
            return false;
        }

        $this->palette[0] = $hex;  // this is the base color, with no change.

        // Here we have a loop which creates 9 shades that are each ten percent
        // lighter than the base color. Each is then added to the array.

        for($i = 0.5; $i < 10; $i += .5)
        {
            $key = 100-($i *10);
            $this->palette['-'.$key] = $this->lighten($hex, $i*.1);
        }

        for($i = 9.5; $i > 0; $i -= .5)
        {
            $key = 100-($i *10);
            $this->palette['+'.$key] = $this->darken($hex, $i*.1);
        }

    } // setPalette()

/*  =========================================================================
    FUNCTION: lighten()
    ========================================================================= */
    function lighten($hex, $percent)
    {
        return $this->mix($hex, $percent, 255);
    } // lighten()

/*  =========================================================================
    FUNCTION: darken()
    ========================================================================= */
    function darken($hex, $percent)
    {
        return $this->mix($hex, $percent, 0);
    } // darken()

/*  =========================================================================
    FUNCTION: mix()
    ========================================================================= */
    function mix($hex, $percent, $mask)
    {
        // Make sure inputs are valid
        if (!is_numeric($percent) || $percent < 0 || $percent > 1) {
            $this->raiseError('percent=' . $percent . 'is not valid',
                              __FUNCTION__, __LINE__);
            return false;
        }

        if (!is_int($mask) || $mask < 0 || $mask > 255) {
            $this->raiseError('mask=' . $mask . ' is not valid',
                              __FUNCTION__, __LINE__);
            return false;
        }

        $rgb = $this->hex2RGB($hex);

        if (!is_array($rgb)) return false;

        for ($i=0; $i<3; $i++)
	{
            $rgb[$i] = round($rgb[$i] * $percent) + round($mask * (1-$percent));

            // In case rounding up causes us to go to 256
            if ($rgb[$i] > 255) $rgb[$i] = 255;

        }

        return $this->RGB2Hex($rgb);
    } // mix()

/*  =========================================================================
    FUNCTION: hex2RGB()
    ========================================================================= */
    function hex2RGB($hex)
    {
        $d = '[a-fA-F0-9]';

        // Make sure $hex is valid
        if (preg_match("/^($d$d)($d$d)($d$d)\$/", $hex, $rgb)) {
            return array(
                hexdec($rgb[1]),
                hexdec($rgb[2]),
                hexdec($rgb[3])
            );
        }

        if (preg_match("/^($d)($d)($d)$/", $hex, $rgb)) {
            return array(
                hexdec($rgb[1] . $rgb[1]),
                hexdec($rgb[2] . $rgb[2]),
                hexdec($rgb[3] . $rgb[3])
            );
        }

        $this->raiseError('cannot convert hex ' . $hex . ' to RGB',
                          __FUNCTION__, __LINE__);
        return false;
    } // hex2RGB()

/*  =========================================================================
    FUNCTION: RGB2Hex()
    ========================================================================= */
    function RGB2Hex($rgb)
    {
        if(!$this->isRGB($rgb)) {
            $this->raiseError('RGB value is not valid', __FUNCTION__, __LINE__);
            return false;
        }

        $hex = '';
        for($i=0; $i < 3; $i++) {

            // Convert the decimal digit to hex
            $hexDigit = dechex($rgb[$i]);

            // Add a leading zero if necessary
            if(strlen($hexDigit) == 1) {
                $hexDigit = "0" . $hexDigit;
            }

            // Append to the hex string
            $hex .= $hexDigit;
        }

        // Return the complete hex string
        return $hex;
    } // RGB2Hex()

/*  =========================================================================
    FUNCTION: isHex()
    ========================================================================= */
    function isHex($hex)
    {
        // Regexp for a valid hex digit
        $d = '[a-fA-F0-9]';

        // Make sure $hex is valid
        if (preg_match("/^#?$d$d$d$d$d$d\$/", $hex) ||
            preg_match("/^#?$d$d$d\$/", $hex)) {
            return true;
        }
        return false;
    } // isHex()

/*  =========================================================================
    FUNCTION: isRGB()
    ========================================================================= */
    function isRGB($rgb)
    {
        // Returns true if $rgb is an array with three valid
        // decimal color digits.

        if (!is_array($rgb) || count($rgb) != 3) {
            return false;
        }

        for($i=0; $i < 3; $i++) {

            // Get the decimal digit
            $dec = intval($rgb[$i]);

            // Make sure the decimal digit is between 0 and 255
            if (!is_int($dec) || $dec < 0 || $dec > 255) {
                return false;
            }
        }

        return true;
    } //isRGB()

/*  =========================================================================
    FUNCTION: raiseError()
    ========================================================================= */
    function raiseError($message,$function, $line)
    {
        print $message;
        print ' [function: '.$function.', line: '.$line.']<br />';
    } // raiseError();

}
?>